/*
 * ALSA driver for Panasonic UniPhier series.
 * 
 * Copyright (c) 2013 Panasonic corporation.
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; version 2
 * of the License.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 */

#include <linux/module.h>
#include <linux/moduleparam.h>
#include <linux/init.h>

#include <linux/kernel.h>
#include <linux/errno.h>
#include <linux/types.h>
#include <linux/spinlock.h>
#include <linux/vmalloc.h>
#include <linux/io.h>
#include <linux/fs.h>
#include <linux/fcntl.h>
#include <linux/sched.h>
#include <linux/signal.h>
#include <linux/wait.h>
#include <linux/poll.h>
#include <linux/kthread.h>
#include <linux/delay.h>
#include <linux/cdev.h>
#include <linux/proc_fs.h>
#include <linux/platform_device.h>
#include <linux/dma-mapping.h>

#include "mn2ws-pcm.h"

MODULE_AUTHOR("Katsuhiro Suzuki <suzuki.katsuhiro002@jp.panasonic.com>");
MODULE_DESCRIPTION("Panasonic UniPhier PCM Driver");
MODULE_LICENSE("GPL");

//devices
int mn2ws_pcm_args_devs = 8;
module_param(mn2ws_pcm_args_devs, int, S_IRUGO);

static int mn2ws_pcm_probe(struct platform_device *device);
static int mn2ws_pcm_remove(struct platform_device *device);
static void mn2ws_pcm_shutdown(struct platform_device *device);
static int mn2ws_pcm_suspend(struct platform_device *device, pm_message_t state);
static int mn2ws_pcm_resume(struct platform_device *device);

static int mn2ws_pcm_dev_free(struct snd_device *device);

static int mn2ws_pcm_proc_read(char *buf, char **start, off_t offset, 
	int count, int *eof, void *data);


struct platform_driver mn2ws_pcm_plat_driver = {
	.driver = {
		.name = "(initialize later)", 
	}, 
	.probe    = mn2ws_pcm_probe, 
	.remove   = mn2ws_pcm_remove, 
	.shutdown = mn2ws_pcm_shutdown, 
	.suspend  = mn2ws_pcm_suspend, 
	.resume   = mn2ws_pcm_resume, 
};

static struct snd_device_ops mn2ws_pcm_dev_ops = {
	.dev_free  = mn2ws_pcm_dev_free, 
	//.dev_register
	//.dev_disconnect
};


static int mn2ws_pcm_devnum = 0;
static struct mn2ws_pcm_dev *mn2ws_pcm_devices = NULL;

static struct platform_device **mn2ws_pcm_plat_devs = NULL;

static struct proc_dir_entry *mn2ws_pcm_procent = NULL;

//for test mode
int mn2ws_pcm_test_mode = 0;




static int mn2ws_pcm_probe(struct platform_device *device)
{
	struct mn2ws_pcm_dev *d = 
		*((struct mn2ws_pcm_dev **)device->dev.platform_data);
	struct snd_card *card = NULL;
	struct mn2ws_pcm_chip *chip = NULL;
	struct snd_pcm *pcm = NULL;
	struct snd_hwdep *hwdep = NULL;
	int play_devs, cap_devs;
	int result = -ENXIO;
	int i;
	
	//DPRINTF("%s\n", __func__);
	
	//save the pointer of platform device
	d->plat_dev = device;
	
	if (down_interruptible(&d->sem_dev)) {
		PRINTF_WARN("cannot down semaphore.\n");
		return 0;
	}
	
	//1) create sound 'card'
	for (i = 0; i < SNDRV_CARDS; i++) {
		d->card_index = SNDRV_DEFAULT_IDX1;
		memset(d->card_xid, 0, sizeof(d->card_xid));
		snprintf(d->card_xid, sizeof(d->card_xid) - 1, 
			"%s%d", "mn2ws", d->ch);
		
		//create 'card'
		result = snd_card_create(d->card_index, d->card_xid, 
			THIS_MODULE, 0, &card);
		if (result < 0) {
			//try next id
			continue;
		}
		
		break;
	}
	if (i == SNDRV_CARDS) {
		PRINTF_WARN("cannot create sound card.\n");
		result = -ENOENT;
		goto err_out1;
	}
	
	//snd_component_add(card, "hoge");
	//snd_card_set_dev(card, &dev->dev);
	
	
	//2) create sound 'chip'
	chip = kzalloc(sizeof(struct mn2ws_pcm_chip), GFP_KERNEL);
	if (chip == NULL) {
		PRINTF_WARN("cannot allocate chip %d.\n", 
			d->ch);
		result = -ENOENT;
		goto err_out2;
	}
	
	//link chip -> card, card -> chip, chip -> device
	chip->card = card;
	card->private_data = chip;
	chip->device = d;
	
	
	//set driver ID and name
	snprintf(card->driver, sizeof(card->driver) - 1, 
		"UniPhier PCM");
	snprintf(card->shortname, sizeof(card->shortname) - 1, 
		"Panasonic %s", get_pcm_dev_name());
	snprintf(card->longname, sizeof(card->longname) - 1, 
		"Panasonic Uniphier %s at 0x%08x irq %i",
		get_pcm_dev_name(), d->play.paddr_buf, -1/*chip->irq*/);
	
	//create sound component
	result = snd_device_new(card, SNDRV_DEV_LOWLEVEL, 
		chip, &mn2ws_pcm_dev_ops);
	if (result < 0) {
		PRINTF_WARN("cannot register sound component %d.\n", 
			d->ch);
		result = -ENOENT;
		goto err_out2;
	}
	
	
	//3) register pcm playback/capture interfaces
	play_devs = 0;
	if (d->play.desc->enabled) {
		play_devs = 1;
	}
	cap_devs = 0;
	if (d->cap.desc->enabled) {
		cap_devs = 1;
	}
	result = snd_pcm_new(card, "", 
		0, play_devs, cap_devs, &pcm);
	if (result < 0) {
		PRINTF_WARN("cannot create pcm interfaces %d.\n", 
			d->ch);
		result = -ENOENT;
		goto err_out3;
	}
	
	//link pcm -> chip, chip -> pcm
	pcm->private_data = chip;
	chip->pcm = pcm;
	
	//set pcm name
	strcpy(pcm->name, get_pcm_dev_name());
	strcat(pcm->name, " pcm");
	//set pcm operations
	if (d->play.desc->enabled) {
		snd_pcm_set_ops(pcm, SNDRV_PCM_STREAM_PLAYBACK, 
			&mn2ws_pcm_playback_ops);
	}
	if (d->cap.desc->enabled) {
		snd_pcm_set_ops(pcm, SNDRV_PCM_STREAM_CAPTURE, 
			&mn2ws_pcm_capture_ops);
	}
	
	
	//4) register hwdep interfaces
	result = snd_hwdep_new(card, "", 
		0, &hwdep);
	if (result < 0) {
		PRINTF_WARN("cannot create hwdep interfaces %d.\n", 
			d->ch);
		result = -ENOENT;
		goto err_out3;
	}
	
	//link pcm -> chip, chip -> pcm
	hwdep->private_data = chip;
	chip->hwdep = hwdep;
	
	//set hwdep name
	strcpy(hwdep->name, get_pcm_dev_name());
	strcat(hwdep->name, " hwdep");
	
	//set hwdep operations
	if (d->hwd.desc->enabled) {
		memmove(&hwdep->ops, &mn2ws_pcm_hwdep_ops, 
			sizeof(mn2ws_pcm_hwdep_ops));
	}
	
	
	//5) register control interfaces
	for (i = 0; i < mn2ws_pcm_mixer_num; i++) {
		result = snd_ctl_add(card, snd_ctl_new1(
			&mn2ws_pcm_mixer_ops[i], chip));
		if (result < 0) {
			PRINTF_WARN("cannot create control(mixer) "
				"interfaces %d.\n", d->ch);
			result = -ENOENT;
			goto err_out3;
		}
	}
	
	
	//link device -> card, device -> chip, 
	//     device -> pcm, device -> hwdep
	d->card = card;
	d->chip = chip;
	d->pcm = pcm;
	d->hwdep = hwdep;
	
	
	//finally, register sound card
	result = snd_card_register(card);
	if (result < 0) {
		//cannot register sound card
		PRINTF_WARN("cannot register sound card %d.\n", 
			d->ch);
		result = -ENOENT;
		goto err_out3;
	}
	
	PRINTF_NOTE("probe pcm dev-%d(card:%d).\n", 
		device->id, d->ch);
	
	up(&d->sem_dev);
	
	return 0;
	
err_out3:
	if (chip) {
		kfree(chip);
		chip = NULL;
	}
	
err_out2:
	if (card) {
		snd_card_free(card);
		card = NULL;
	}
	
err_out1:
	up(&d->sem_dev);
	
	return result;
}

static int mn2ws_pcm_remove(struct platform_device *device)
{
	struct mn2ws_pcm_dev *d = 
		*((struct mn2ws_pcm_dev **)device->dev.platform_data);
	
	DPRINTF("%s\n", __func__);
	
	if (d->chip) {
		kfree(d->chip);
		d->chip = NULL;
	}
	
	if (d->card) {
		snd_card_free(d->card);
		d->card = NULL;
	}
	
	return 0;
}

static void mn2ws_pcm_shutdown(struct platform_device *device)
{
	DPRINTF("%s\n", __func__);
}

static int mn2ws_pcm_suspend(struct platform_device *device, pm_message_t state)
{
	DPRINTF("%s\n", __func__);
	
	return 0;
}

static int mn2ws_pcm_resume(struct platform_device *device)
{
	DPRINTF("%s\n", __func__);
	
	return 0;
}




static int mn2ws_pcm_dev_free(struct snd_device *device)
{
	DPRINTF("%s\n", __func__);
	
	return 0;
}




static int mn2ws_pcm_proc_read(char *buf, char **start, off_t offset, 
	int count, int *eof, void *data)
{
	struct mn2ws_pcm_dev *d;
	int len;
	int i;
	
	len = 0;
	
	for (i = 0; i < mn2ws_pcm_devnum; i++) {
		d = &mn2ws_pcm_devices[i];
		
		len += sprintf(buf + len, 
			"ch%d_p: paddr:0x%08x, map:0x%p\n", 
			d->ch, d->play.paddr_buf, 
			d->play.buf);
		len += sprintf(buf + len, 
			"       hw(r:%05x,w:%05x,rem:%05x) "
			"alsa(r:%05x,w:%05x,rem:%05x)\n", 
			(int)get_rp_ringbuf(&d->play.hw), 
			(int)get_wp_ringbuf(&d->play.hw), 
			(int)get_remain_ringbuf(&d->play.hw), 
			(int)get_rp_ringbuf(&d->play.alsa), 
			(int)get_wp_ringbuf(&d->play.alsa), 
			(int)get_remain_ringbuf(&d->play.alsa));
		
		len += sprintf(buf + len, 
			"ch%d_c: paddr:0x%08x, map:0x%p\n", 
			d->ch, d->cap.paddr_buf, 
			d->cap.buf);
		len += sprintf(buf + len, 
			"       hw(r:%05x,w:%05x,rem:%05x) "
			"alsa(r:%05x,w:%05x,rem:%05x)\n", 
			(int)get_rp_ringbuf(&d->cap.hw), 
			(int)get_wp_ringbuf(&d->cap.hw), 
			(int)get_remain_ringbuf(&d->cap.hw), 
			(int)get_rp_ringbuf(&d->cap.alsa), 
			(int)get_wp_ringbuf(&d->cap.alsa), 
			(int)get_remain_ringbuf(&d->cap.alsa));
	}
	
	return len;
}

static int mn2ws_pcm_proc_write(struct file *filp, const char *buf, 
	unsigned long len, void *data)
{
	#define BUF_LEN    64
	static char work_buf[BUF_LEN];
	int i;
	
	if (len >= BUF_LEN){
		PRINTF_WARN("proc_write len = %ld\n", len);
		return -ENOSPC;
	}
	if (copy_from_user(work_buf, buf, len)) {
		return -EFAULT;
	}
	work_buf[len] = '\0';

	//erase \n and \r
	for (i = 0; i < len; i++) {
		if (work_buf[i] == '\n' || work_buf[i] == '\r') {
			work_buf[i] = '\0';
		}
	}
	DPRINTF(KERN_INFO "proc_write '%s'\n", work_buf);
	
	//do debug commands
	if (strcasecmp("test_on", work_buf) == 0) {
		//test mode on
		mn2ws_pcm_test_mode = 1;
	} else if (strcasecmp("test_off", work_buf) == 0) {
		//test mode off
		mn2ws_pcm_test_mode = 0;
	}
	
	PRINTF_NOTE("test mode: mode:%-3s\n", 
		mn2ws_pcm_test_mode ? "ON":"OFF");
	
	return len;
}

/**
 * ϡɥޤ
 * 
 * @param desc PCM ϡɥεһ
 * @return  0ԤΥ顼
 */
int init_pcm_generic(struct mn2ws_pcm_desc *desc)
{
	memset(desc, 0, sizeof(struct mn2ws_pcm_desc));
	
	return 0;
}

/**
 * ϡɥ˴ޤ
 * 
 * @param desc PCM ϡɥεһ
 * @return  0ԤΥ顼
 */
int term_pcm_generic(struct mn2ws_pcm_desc *desc)
{
	return 0;
}

/**
 * ⥸塼ζ̽Ԥޤ
 * 
 * @param descs PCM ϡɥεһҤ
 * @return  0ԤΥ顼
 */
int mn2ws_pcm_init_module(struct mn2ws_pcm_desc descs[])
{
	struct mn2ws_pcm_dev *dev;
	struct platform_device *plat_dev = NULL;
	struct resource *resource_reg;
	int result = -ENXIO;
	int i, j;
	
	//check the arguments
	if (mn2ws_pcm_args_devs <= 0) {
		PRINTF_WARN("invalid param %s=%d.\n", 
			"pcm_args_devs", mn2ws_pcm_args_devs);
		result = -EINVAL;
		goto err_out1;
	}
	if (mn2ws_pcm_args_devs > get_pcm_dev_count()) {
		PRINTF_WARN("too many devices specified(%s=%d)."
			" fix to %d.\n", 
			"pcm_args_devs", mn2ws_pcm_args_devs, 
			get_pcm_dev_count());
	}
	mn2ws_pcm_devnum = min(mn2ws_pcm_args_devs, get_pcm_dev_count());
	
	//allocate device info
	mn2ws_pcm_devices = kzalloc(
		mn2ws_pcm_devnum * sizeof(struct mn2ws_pcm_dev), 
		GFP_KERNEL);
	if (mn2ws_pcm_devices == NULL) {
		PRINTF_WARN("failed to kmalloc pcm_devices.\n");
		result = -ENOMEM;
		goto err_out1;
	}
	
	//initialize device info
	for (i = 0; i < mn2ws_pcm_devnum; i++) {
		dev = &mn2ws_pcm_devices[i];
		
		dev->ch = i;
		dev->desc = &descs[i];
		dev->hwd.desc = &descs[i].hwd;
		dev->play.desc = &descs[i].play;
		dev->cap.desc = &descs[i].cap;
		
		dev->play.type = MN2WS_PCMIF_PLAYBACK;
		dev->play.thread_main = mn2ws_pcm_play_thread;
		dev->cap.type = MN2WS_PCMIF_CAPTURE;
		dev->cap.thread_main = mn2ws_pcm_cap_thread;
		
		//initialize semaphore, spinlock, wait_queue
		sema_init(&dev->sem_dev, 1);
		
		//mmap register areas
		dev->map_regs = kzalloc(
			dev->desc->n_res_regs * sizeof(struct mem_mapping), 
			GFP_KERNEL);
		if (dev->map_regs == NULL) {
			PRINTF_WARN("ch%d: failed to kmalloc map_regs.\n", 
				dev->ch);
			result = -ENOMEM;
			goto err_out3;
		}
		for (j = 0; j < dev->desc->n_res_regs; j++) {
			resource_reg = &dev->desc->res_regs[j];
			
			result = pcm_mmap(&dev->map_regs[j], 
				resource_reg->end - resource_reg->start, 
				resource_reg->start);
			if (result != 0) {
				PRINTF_WARN("ch%d-%d: failed to map registers, "
					"error %d.\n", 
					dev->ch, j, result);
				result = -ENOMEM;
				goto err_out3;
			}
		}
		
		result = mn2ws_pcm_hwdep_init(dev);
		if (result != 0) {
			PRINTF_WARN("ch%d_h: failed to init, error %d.\n", 
				dev->ch, result);
			result = -EFAULT;
			goto err_out3;
		}
		
		result = mn2ws_pcm_pcmif_init(dev, &dev->play);
		if (result != 0) {
			PRINTF_WARN("ch%d_p: failed to init, error %d.\n", 
				dev->ch, result);
			result = -EFAULT;
			goto err_out3;
		}
	
		result = mn2ws_pcm_pcmif_init(dev, &dev->cap);
		if (result != 0) {
			PRINTF_WARN("ch%d_c: failed to init, error %d.\n", 
				dev->ch, result);
			result = -EFAULT;
			goto err_out3;
		}
	}
	
	
	//allocate devices of mn2ws_pcm for platform bus
	mn2ws_pcm_plat_devs = kzalloc(
		mn2ws_pcm_devnum * sizeof(struct platform_device *), 
		GFP_KERNEL);
	if (mn2ws_pcm_plat_devs == NULL) {
		PRINTF_WARN("failed to kmalloc pcm_plat_devs.\n");
		result = -ENOMEM;
		goto err_out3;
	}
	
	//register devices of mn2ws_pcm for platform bus
	for (i = 0; i < mn2ws_pcm_devnum; i++) {
		dev = &mn2ws_pcm_devices[i];
		plat_dev = platform_device_alloc(get_pcm_dev_name(), i);
		if (plat_dev == NULL) {
			PRINTF_WARN("ch%d: platform_device_alloc() failed.\n", 
				dev->ch);
			result = -ENOMEM;
			goto err_out5;
		}
		
		mn2ws_pcm_plat_devs[i] = plat_dev;
		
		//device info
		platform_device_add_resources(plat_dev, 
			dev->desc->res_regs, dev->desc->n_res_regs);
		platform_device_add_data(plat_dev, 
			&dev, sizeof(struct mn2ws_pcm_dev **));
		
		result = platform_device_add(plat_dev);
		if (result < 0) {
			PRINTF_WARN("platform_device_add() failed.\n");
			goto err_out5;
		}
	}
	
	//register the driver of mn2ws_pcm for platform bus
	mn2ws_pcm_plat_driver.driver.name = get_pcm_dev_name();
	result = platform_driver_register(&mn2ws_pcm_plat_driver);
	if (result < 0) {
		PRINTF_WARN("platform_driver_register() failed.\n");
		goto err_out5;
	}
	
	
	//register the entry of /proc
	mn2ws_pcm_procent = create_proc_entry(MN2WS_PCM_PROC_ENTRY, 0, NULL);
	if (mn2ws_pcm_procent == NULL) {
		PRINTF_WARN("failed to create_proc_read_entry, "
			"error %d\n", 
			(int)mn2ws_pcm_procent);
		goto err_out6;
	}
	mn2ws_pcm_procent->read_proc = mn2ws_pcm_proc_read;
	mn2ws_pcm_procent->write_proc = mn2ws_pcm_proc_write;
	
	//success
	PRINTF_NOTE("v0.10.94 loaded\n");
	
	return 0;
	
//err_out7:
	remove_proc_entry(MN2WS_PCM_PROC_ENTRY, NULL);

err_out6:
	platform_driver_unregister(&mn2ws_pcm_plat_driver);
	
err_out5:
	for (i = 0; i < mn2ws_pcm_devnum; i++) {
		platform_device_del(mn2ws_pcm_plat_devs[i]);
	}
	
//err_out4:
	kfree(mn2ws_pcm_plat_devs);
	mn2ws_pcm_plat_devs = NULL;
	
err_out3:
	for (i = 0; i < mn2ws_pcm_devnum; i++) {
		dev = &mn2ws_pcm_devices[i];
		
		result = mn2ws_pcm_pcmif_exit(dev, &dev->cap);
		if (result != 0) {
			PRINTF_WARN("ch%d_c: failed to exit, error %d.\n", 
				dev->ch, result);
		}
		
		result = mn2ws_pcm_pcmif_exit(dev, &dev->play);
		if (result != 0) {
			PRINTF_WARN("ch%d_p: failed to exit, error %d.\n", 
				dev->ch, result);
		}
		
		result = mn2ws_pcm_hwdep_exit(dev);
		if (result != 0) {
			PRINTF_WARN("ch%d_h: failed to exit, error %d.\n", 
				dev->ch, result);
		}
		
		for (j = 0; j < dev->desc->n_res_regs; j++) {
			pcm_munmap(&dev->map_regs[j]);
		}
		kfree(dev->map_regs);
		dev->map_regs = NULL;
	}
	
//err_out2:
	kfree(mn2ws_pcm_devices);
	mn2ws_pcm_devices = NULL;
	
err_out1:
	return result;
}

/**
 * ⥸塼ζ̽Ԥޤ
 * 
 * @param descs PCM ϡɥεһҤ
 * @return  0ԤΥ顼
 */
int mn2ws_pcm_exit_module(struct mn2ws_pcm_desc descs[])
{
	struct mn2ws_pcm_dev *dev;
	int result, i, j;
	
	remove_proc_entry(MN2WS_PCM_PROC_ENTRY, NULL);
	
	platform_driver_unregister(&mn2ws_pcm_plat_driver);
	
	for (i = 0; i < mn2ws_pcm_devnum; i++) {
		platform_device_del(mn2ws_pcm_plat_devs[i]);
	}
	
	kfree(mn2ws_pcm_plat_devs);
	mn2ws_pcm_plat_devs = NULL;
	
	for (i = 0; i < mn2ws_pcm_devnum; i++) {
		dev = &mn2ws_pcm_devices[i];
		
		result = mn2ws_pcm_pcmif_exit(dev, &dev->cap);
		if (result != 0) {
			PRINTF_WARN("ch%d_c: failed to exit, error %d.\n", 
				dev->ch, result);
		}
		
		result = mn2ws_pcm_pcmif_exit(dev, &dev->play);
		if (result != 0) {
			PRINTF_WARN("ch%d_p: failed to exit, error %d.\n", 
				dev->ch, result);
		}
		
		result = mn2ws_pcm_hwdep_exit(dev);
		if (result != 0) {
			PRINTF_WARN("ch%d_h: failed to exit, error %d.\n", 
				dev->ch, result);
		}
		
		for (j = 0; j < dev->desc->n_res_regs; j++) {
			pcm_munmap(&dev->map_regs[j]);
		}
		kfree(dev->map_regs);
		dev->map_regs = NULL;
	}
	
	kfree(mn2ws_pcm_devices);
	mn2ws_pcm_devices = NULL;
	
	//success
	PRINTF_NOTE("unloaded\n");
	
	return 0;
}
